-- Author: U_BMP
-- Group: https://vk.com/biomodprod_utilit_fs
-- Date: 19.11.2025

MudWetnessMiniHUD = {}
MudWetnessMiniHUD.name = g_currentModName or "MudWetnessMiniHUD"
MudWetnessMiniHUD.path = g_currentModDirectory or ""

addModEventListener(MudWetnessMiniHUD)

-- =========================================================
-- SETTINGS
-- =========================================================

MudWetnessMiniHUD.enabled = true
MudWetnessMiniHUD.debug   = false

MudWetnessMiniHUD.wetIconRel  = "gui/wet.dds"
MudWetnessMiniHUD.rainIconRel = "gui/rain.dds"

-- icon preset (1..3)
MudWetnessMiniHUD.iconPreset = 1
-- runtime: what preset was applied to overlays (for delayed apply after save reload)
MudWetnessMiniHUD._appliedIconPreset = nil


MudWetnessMiniHUD.ICON_PRESETS = {
    [1] = { wet="gui/wet.dds",        rain="gui/rain.dds" },         -- standard
    [2] = { wet="gui/wet_rmg.dds",    rain="gui/rain_rmg.dds" },     -- RMG
    [3] = { wet="gui/wet_custom.dds", rain="gui/rain_custom.dds" }   -- custom
}


MudWetnessMiniHUD.iconSizePx     = 17
MudWetnessMiniHUD.textPadPx      = 5
MudWetnessMiniHUD.roundToPercent = true

-- transparency
MudWetnessMiniHUD.iconAlpha = 0.85
MudWetnessMiniHUD.textAlpha = 0.85

-- GameInfoDisplay
MudWetnessMiniHUD.gapAboveGameInfoPx = 6

-- manual offsets (pixels)
-- offsetXPx: + -> left,  - -> right
-- offsetYPx: + -> up,    - -> down
MudWetnessMiniHUD.offsetXPx = 15
MudWetnessMiniHUD.offsetYPx = 0

-- background
MudWetnessMiniHUD.bgEnable = true
MudWetnessMiniHUD.bgAlpha  = 0.45

-- =========================================================
-- UTILS
-- =========================================================

local function clamp(v, a, b)
    if v < a then return a end
    if v > b then return b end
    return v
end

local function getWeather()
    if g_currentMission ~= nil
        and g_currentMission.environment ~= nil
        and g_currentMission.environment.weather ~= nil then
        return g_currentMission.environment.weather
    end
    return nil
end

local function getWetAndRain()
    local weather = getWeather()
    if weather == nil then
        return 0, 0, 0, 0
    end

    local wet = 0
    if weather.getGroundWetness ~= nil then
        wet = weather:getGroundWetness() or 0
    end

    local rain = 0
    if weather.getRainFallScale ~= nil then
        rain = weather:getRainFallScale() or 0
    end

    local snow = 0
    if weather.getSnowFallScale ~= nil then
        snow = weather:getSnowFallScale() or 0
    end

    -- we want "precipitation intensity" to show on rain icon:
    -- rain OR snow (whatever is active / stronger)
    local precip = math.max(rain, snow)

    -- return 4 values (extra ones are ignored where not needed)
    return clamp(wet, 0, 1), clamp(precip, 0, 1), clamp(rain, 0, 1), clamp(snow, 0, 1)
end


local function fmt01(v, asPercent)
    if asPercent then
        return string.format("%d%%", math.floor(v * 100 + 0.5))
    end
    return string.format("%.2f", v)
end

-- =========================================================
-- LIFECYCLE
-- =========================================================

function MudWetnessMiniHUD:loadMap()
    -- Build overlays once with current value (may still be default at this moment)
    self._appliedIconPreset = -1
    self:setIconPreset(self.iconPreset or 1, true, true)

    addConsoleCommand("gsMudWetMini", "MudWetnessMiniHUD on/off/toggle/debug", "consoleCommand", self)
end



function MudWetnessMiniHUD:deleteMap()
    removeConsoleCommand("gsMudWetMini")
    self:_deleteOverlays()
end

function MudWetnessMiniHUD:consoleCommand(arg)
    if arg == "on" then
        self.enabled = true
    elseif arg == "off" then
        self.enabled = false
    elseif arg == "toggle" or arg == nil or arg == "" then
        self.enabled = not self.enabled
    elseif arg == "debug" then
        self.debug = not self.debug
    end

    local wet, precip, rain, snow = getWetAndRain()
	return string.format(
		"MudWetnessMiniHUD enabled=%s debug=%s wet=%s precip=%s (rain=%s snow=%s)",
		tostring(self.enabled), tostring(self.debug),
		fmt01(wet, true), fmt01(precip, true),
		fmt01(rain, true), fmt01(snow, true)
	)

end

function MudWetnessMiniHUD:_deleteOverlays()
    if self.wetIcon ~= nil then self.wetIcon:delete(); self.wetIcon = nil end
    if self.rainIcon ~= nil then self.rainIcon:delete(); self.rainIcon = nil end
    if self.bgOverlay ~= nil then self.bgOverlay:delete(); self.bgOverlay = nil end
end

function MudWetnessMiniHUD:_rebuildOverlays()
    self:_deleteOverlays()

    self.wetIcon  = Overlay.new(self.wetIconPath,  0, 0, 0.01, 0.01)
    self.rainIcon = Overlay.new(self.rainIconPath, 0, 0, 0.01, 0.01)

    if self.bgEnable then
        self.bgOverlay = Overlay.new(nil, 0, 0, 0.01, 0.01)
        self.bgOverlay:setColor(0, 0, 0, self.bgAlpha or 0.45)
    end
end

function MudWetnessMiniHUD:setIconPreset(presetIndex, silent, force)
    presetIndex = math.floor(tonumber(presetIndex) or 1)
    if presetIndex < 1 then presetIndex = 1 end
    if presetIndex > 3 then presetIndex = 3 end

    local preset = self.ICON_PRESETS[presetIndex] or self.ICON_PRESETS[1]

    -- early-out only if EVERYTHING already matches (and not forced)
    if not force then
        if self.iconPreset == presetIndex
            and self.wetIconRel  == preset.wet
            and self.rainIconRel == preset.rain
            and self.wetIconPath ~= nil
            and self.rainIconPath ~= nil then
            return
        end
    end

    self.iconPreset = presetIndex

    self.wetIconRel  = preset.wet
    self.rainIconRel = preset.rain

    self.wetIconPath  = Utils.getFilename(self.wetIconRel,  self.path)
    self.rainIconPath = Utils.getFilename(self.rainIconRel, self.path)

    self:_rebuildOverlays()

    if self.debug and not silent then
        print(string.format("[MudWetnessMiniHUD] iconPreset=%d wet=%s rain=%s",
            presetIndex, tostring(self.wetIconRel), tostring(self.rainIconRel)))
    end
end


-- Called by MudSystemSettings after any setting apply (target:onSettingsChanged)
function MudWetnessMiniHUD:onSettingsChanged()
    -- IMPORTANT: MudSystemSettings already updated self.iconPreset before calling us,
    -- so we must FORCE rebuild
    self:setIconPreset(self.iconPreset, true, true)
end



-- =========================================================
-- DRAW
-- =========================================================

function MudWetnessMiniHUD:draw()
    if not self.enabled or g_currentMission == nil or g_currentMission.hud == nil then
        return
    end

    local hud = g_currentMission.hud

    if hud.getIsVisible ~= nil then
        if not hud:getIsVisible() then return end
    elseif hud.isVisible ~= nil then
        if not hud.isVisible then return end
    end

    local gid = hud.gameInfoDisplay
    if gid == nil or gid.getPosition == nil then
        return
    end
	
	if self._appliedIconPreset ~= self.iconPreset then
        self:setIconPreset(self.iconPreset or 1, true, true)
        self._appliedIconPreset = self.iconPreset
    end

    if gid.getVisible ~= nil and not gid:getVisible() then
        return
    end

    local posX, posY = gid:getPosition()

    local sW = gid.scalePixelToScreenWidth ~= nil
        and function(px) return gid:scalePixelToScreenWidth(px) end
        or function(px) return px / 1920 end

    local sH = gid.scalePixelToScreenHeight ~= nil
        and function(px) return gid:scalePixelToScreenHeight(px) end
        or function(px) return px / 1080 end

    local iconW = sW(self.iconSizePx)
    local iconH = sH(self.iconSizePx)

    local padX = sW(self.textPadPx)
    local gapX = sW(10)
    local gapY = sH(self.gapAboveGameInfoPx)

    local offX = sW(self.offsetXPx)
    local offY = sH(self.offsetYPx)

    local textSize = gid.clockTextSize or gid.moneyTextSize or sH(17)

    local wet, rain = getWetAndRain()
    local wetText  = fmt01(wet,  self.roundToPercent)
    local rainText = fmt01(rain, self.roundToPercent)

    setTextAlignment(RenderText.ALIGN_LEFT)
    setTextBold(true)
    setTextColor(1, 1, 1, self.textAlpha)

    local wetW  = iconW + padX + getTextWidth(textSize, wetText)
    local rainW = iconW + padX + getTextWidth(textSize, rainText)

    local blockW = wetW + gapX + rainW
    local blockH = iconH

    local xRight = posX - offX
    local x0 = xRight - blockW
    local y0 = (posY + gapY) + offY

    if self.bgOverlay ~= nil then
        self.bgOverlay:setPosition(x0 - sW(6), y0 - sH(4))
        self.bgOverlay:setDimension(blockW + sW(12), blockH + sH(8))
        self.bgOverlay:render()
    end

    self.wetIcon:setPosition(x0, y0)
    self.wetIcon:setDimension(iconW, iconH)
    self.wetIcon:setColor(1, 1, 1, self.iconAlpha)
    self.wetIcon:render()

    renderText(x0 + iconW + padX, y0 + sH(2), textSize, wetText)

    local rx = x0 + wetW + gapX

    self.rainIcon:setPosition(rx, y0)
    self.rainIcon:setDimension(iconW, iconH)
    self.rainIcon:setColor(1, 1, 1, self.iconAlpha)
    self.rainIcon:render()

    renderText(rx + iconW + padX, y0 + sH(2), textSize, rainText)

    if self.debug then
        setTextBold(false)
        local _, _, rainDbg, snowDbg = getWetAndRain()
		renderText(x0, y0 - sH(12), sH(12),
			string.format("wet=%.2f precip=%.2f rain=%.2f snow=%.2f", wet, rain, rainDbg, snowDbg))

        setTextBold(true)
    end

    setTextBold(false)
end
